﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Helpers;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Shared.Library;
    using Utilities;
    using Hims.Domain.Configurations;

    using TimeZone = Shared.Library.TimeZone;
    using Hims.Shared.UserModels.Filters;
    using Hims.Shared.UserModels.Slots;
    using Hims.Shared.DataFilters;
    using Hims.Shared.UserModels;
    using Hims.Shared.UserModels.Common;
    using Hims.Shared.UserModels.Pharmacy;
    using Hims.Domain.Entities;
    using Hims.Shared.EntityModels;
    using Hims.Api.Senders;
    using Hims.Api.Models;
    using Hims.Shared.UserModels.Scan.ScanAppointment;
    using Hims.Shared.UserModels.Scan.ScanMachineAvailability;
    using Hims.Shared.UserModels.MasterBill;
    using FilterModel = Shared.UserModels.MasterBill.FilterModel;
    using Hims.Shared.UserModels.Labs;
    using Hims.Shared.UserModels.Labs.Report;
    using Hims.Shared.UserModels.Reports;
  

    /// <inheritdoc />
    /// <summary>
    /// The resources controller.
    /// </summary>
    [Route("api/reports")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ReportController : BaseController
    {
        /// <summary>
        /// The resource services.
        /// </summary>
        private readonly IReportService reportServices;

        /// <summary>
        /// The AES helper.
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <summary>
        /// The amazon s 3 configuration.
        /// </summary>
        private readonly IAmazonS3Configuration amazonS3Configuration;

        /// <summary>
        /// The email sender.
        /// </summary>
        private readonly IEmailSender emailSender;

        /// <summary>
        /// The pharmacy service.
        /// </summary>
        private readonly IResourceService resourceService;

        /// <summary>
        /// The pharmacy service.
        /// </summary>
        private readonly IPharmacyService pharmacyService;

        /// <summary>
        /// The running environment.
        /// </summary>
        private readonly IRunningEnvironment runningEnvironment;
        /// <summary>
        /// The provider location services.
        /// </summary>
        private readonly IProviderLocationService providerLocationServices;
        private readonly IAccountService accountService;
        /// <summary>
        /// The scam machine availability services.
        /// </summary>
        private readonly IScanMachineAvailabilityService scanMachineAvailabilityService;

        /// <inheritdoc />
        public ReportController(IReportService reportServices, IAESHelper aesHelper, IAmazonS3Configuration amazonS3Configuration, IEmailSender emailSender, IResourceService resourceService, IPharmacyService pharmacyService, IRunningEnvironment runningEnvironment, IProviderLocationService providerLocationServices, IAccountService accountService, IScanMachineAvailabilityService scanMachineAvailabilityService)
        {
            this.reportServices = reportServices;
            this.aesHelper = aesHelper;
            this.amazonS3Configuration = amazonS3Configuration;
            this.emailSender = emailSender;
            this.resourceService = resourceService;
            this.pharmacyService = pharmacyService;
            this.runningEnvironment = runningEnvironment;
            this.providerLocationServices = providerLocationServices;
            this.accountService = accountService;
            this.scanMachineAvailabilityService = scanMachineAvailabilityService;
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <param name="header"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("in-patients")]
        [ProducesResponseType(200, Type = typeof(List<Report>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInPatientsAsync([FromBody] PatientFilterModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (PatientFilterModel)EmptyFilter.Handler(model);
                model.LocationId = Convert.ToInt32(header.LocationId);
                var inPatients = await this.reportServices.InPatientsAsync(model);
                foreach (var item in inPatients)
                {
                    DateTime d1 = DateTime.Today + item.AdmissionTime;

                    item.AdmissionTimeString = d1.ToString("hh:mm tt");
                }
                return this.Success(inPatients);
            }
            catch (Exception ex)
            {
                return null;
            }
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("pharmacy-sales-reports")]
        [ProducesResponseType(200, Type = typeof(List<PharamacyReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInPharamctReportAsync([FromBody] PharmacyReportFilterModel model, [FromHeader] LocationHeader location)
        {
            model = (PharmacyReportFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var inPatients = await this.reportServices.FetchInPharamctReportAsync(model);
            return this.Success(inPatients);
        }

        /// <summary>
        /// To fetch list of daily sales.
        /// </summary>
        /// <param name="model">
        /// 
        /// 
        /// The sales model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of sales.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("daily-sales-reports")]
        [ProducesResponseType(200, Type = typeof(List<DailySalesMedicationModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDailySalesReportAsync([FromBody] DailySalesMedicationFilterModel model, [FromHeader] LocationHeader location)
        {
            model = (DailySalesMedicationFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var dailySalesReports = await this.reportServices.FetchDailySalesReportAsync(model);
            return this.Success(dailySalesReports);
        }

        /// <summary>
        /// To fetch list of patient pharmacy.
        /// </summary>
        /// <param name="model">
        /// The patient pharmacy model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patient pharmacy sales.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("patient-pharmacy-reports")]
        [ProducesResponseType(200, Type = typeof(List<PharmacySaleBill>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientPharmacyReportAsync([FromBody] PharmacySaleBill model, [FromHeader] LocationHeader location)
        {
            model = (PharmacySaleBill)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var patientPharmacyReports = await this.reportServices.FetchPatientPharmacyReportAsync(model);
            return this.Success(patientPharmacyReports);
        }

        /// <summary>
        /// To fetch list of pharmacy purchase bills.
        /// </summary>
        /// <param name="model">
        /// The pharmacy purchase model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of pharmacy purchase bills.
        /// - 500 - Problem with Server side code.
        /// </remarks>

        [HttpPost]
        [Authorize]
        [Route("pharmacy-purchase-reports")]
        [ProducesResponseType(200, Type = typeof(List<PharmacyPurchaseReportBill>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPharmacyPurchaseReportAsync([FromBody] PharmacyPurchaseReportBill model, [FromHeader] LocationHeader location)
        {
            model = (PharmacyPurchaseReportBill)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var pharmacyPurchaseReports = await this.reportServices.FetchPharmacyPurchaseReportAsync(model);
            return this.Success(pharmacyPurchaseReports);
        }

        /// <summary>
        /// To fetch list of pharmacy purchase bills.
        /// </summary>
        /// <param name="model">
        /// The pharmacy purchase model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of pharmacy purchase bills.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("product-purchase-reports")]
        [ProducesResponseType(200, Type = typeof(List<PharmacyPurchaseReportBill>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProductPurchaseReportAsync([FromBody] PharmacyPurchaseReportBill model, [FromHeader] LocationHeader location)
        {
            model = (PharmacyPurchaseReportBill)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var pharmacyPurchaseReports = await this.reportServices.FetchProductPurchaseReportAsync(model);
            return this.Success(pharmacyPurchaseReports);
        }

        /// <summary>
        /// To fetch list of pharmacy purchase bills.
        /// </summary>
        /// <param name="model">
        /// The pharmacy purchase model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of pharmacy purchase bills.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("product-sale-reports")]
        [ProducesResponseType(200, Type = typeof(List<PharmacyPurchaseReportBill>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProductSaleReportAsync([FromBody] PharmacyPurchaseReportBill model, [FromHeader] LocationHeader location)
        {
            model = (PharmacyPurchaseReportBill)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var pharmacyPurchaseReports = await this.reportServices.FetchProductSaleReportAsync(model);
            return this.Success(pharmacyPurchaseReports);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("dashboard-in-patients")]
        [ProducesResponseType(200, Type = typeof(List<Report>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDashboardInPatientsAsync([FromBody] PatientFilterModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (PatientFilterModel)EmptyFilter.Handler(model);
                model.LocationId = Convert.ToInt32(header.LocationId);
                var inPatients = await this.reportServices.FetchDashboardInPatientsAsync(model);
                string format = @"hh\:mm";
                foreach (var item in inPatients)
                {
                    item.AdmissionTimeString = item.AdmissionTime.ToString(format);
                }
                return this.Success(inPatients);
            }
            catch (Exception ex)
            {
                return null;
            }
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("out-patients")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchOutPatientsAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model.LocationId = Convert.ToInt32(header.LocationId);
            var outPatients = await this.reportServices.OutPatientsAsync(model);
            foreach (var item in outPatients)
            {
                try
                {
                    //  if (item.AppointmentTime!=)
                    DateTime d1 = DateTime.Today + item.AppointmentTime;
                    item.AppointmentTimeString = d1.ToString("hh:mm tt");
                    //item.TotAmt = Int64.Parse(item.TotalAmount);
                    // int t = Convert.ToInt32(item.TotalAmount.ToString());
                }
                catch (Exception e)
                {
                    return null;
                }
            }
            return this.Success(outPatients);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("dashboard-out-patients")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchOutDashboardPatientsAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var inPatients = await this.reportServices.FetchOutDashboardPatientsAsync(model);

            foreach (var item in inPatients)
            {
                DateTime d1 = DateTime.Today + item.AppointmentTime;

                item.AppointmentTimeString = d1.ToString("hh:mm tt");
            }
            return this.Success(inPatients);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("dashboard-followUp-patients")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchFollowUpDashboardPatientsAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            model.ProviderId = await this.accountService.GetReferenceAsync(model.AccountId);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var inPatients = await this.reportServices.FetchFollowUpDashboardPatientsAsync(model);

            foreach (var item in inPatients)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
            }
            return this.Success(inPatients);
        }
        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("dashboard-online-patients")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchOnlineDashboardPatientsAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            model.ProviderId = await this.accountService.GetReferenceAsync(model.AccountId);
            var inPatients = await this.reportServices.FetchOnlineDashboardPatientsAsync(model);
            foreach (var item in inPatients)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
            }
            return this.Success(inPatients);
        }
        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("dashboard-newpatients-appointmnet")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchNewPatientAptDashboardPatientsAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            model.ProviderId = await this.accountService.GetReferenceAsync(model.AccountId);
            var inPatients = await this.reportServices.FetchNewPatientAppointmentDashboardPatientsAsync(model);
            foreach (var item in inPatients)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
            }
            return this.Success(inPatients);
        }
        [HttpPost]
        [Authorize]
        [Route("dashboard-new-appointment")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchNewAppointmentDashboardPatientsAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            model.ProviderId = await this.accountService.GetReferenceAsync(model.AccountId);
            var inPatients = await this.reportServices.FetchNewApointmentDashboardPatientsAsync(model);
            foreach (var item in inPatients)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
            }
            return this.Success(inPatients);
        }
        [HttpPost]
        [Authorize]
        [Route("dashboard-walkIn-appointmnet")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchWalkInDashboardPatientAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            model.ProviderId = await this.accountService.GetReferenceAsync(model.AccountId);
            var inPatients = await this.reportServices.FetchWalkInDashboardPatientsAsync(model);
            foreach (var item in inPatients)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
            }
            return this.Success(inPatients);
        }
        [HttpPost]
        [Authorize]
        [Route("dashboard-ot-appointmnet")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchOTDashboardPatientAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            model.ProviderId = await this.accountService.GetReferenceAsync(model.AccountId);
            var inPatients = await this.reportServices.FetchOTApointmentDashboardPatientsAsync(model);
            foreach (var item in inPatients)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
            }
            return this.Success(inPatients);
        }
        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("receipt-reports")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchReceptionistReportsAsync([FromBody] ReceiptReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (ReceiptReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchReceptionistReportsAsync(model);
            return this.Success(receiptReports);
        }

        /// <summary>
        /// To fetch list of employee revenue.
        /// </summary>
        /// <param name="model">
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of employee revenue.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("employee-revenue-reports")]
        [ProducesResponseType(200, Type = typeof(List<EmployeeRevenueReportsFilterModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchEmployeeRevenueReportsAsync([FromBody] EmployeeRevenueReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (EmployeeRevenueReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var employeeRevenueReports = await this.reportServices.FetchEmployeeRevenueReportsAsync(model);
            return this.Success(employeeRevenueReports);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("indent-inventory-purchase-bill-reports")]
        [ProducesResponseType(200, Type = typeof(List<InventoryPurchaseBillReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInventoryPurchaseRegisterReportsAsync([FromBody] InventoryPurchaseRegisteryFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (InventoryPurchaseRegisteryFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var reports = await this.reportServices.FetchInventoryPurchaseRegisterReportsAsync(model);
            return this.Success(reports);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("indent-inventory-stock-reports")]
        [ProducesResponseType(200, Type = typeof(List<InventoryStockReportModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInventoryStockReportsAsync([FromBody] InventoryStockReportFilterModel model)
        {
            model = (InventoryStockReportFilterModel)EmptyFilter.Handler(model);
            var reports = await this.reportServices.FetchInventoryStockReportsAsync(model);
            return this.Success(reports);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("indent-inventory-reports")]
        [ProducesResponseType(200, Type = typeof(List<IndentInventoryModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchIndentInventoryReportsAsync([FromBody] IndentInventoryFilterModel model)
        {
            model = (IndentInventoryFilterModel)EmptyFilter.Handler(model);
            var receiptReports = await this.reportServices.FetchIndentInventoryReportsAsync(model);
            return this.Success(receiptReports);
        }

        /// <summary>
        /// The update sale bill quantity async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        //[HttpPost]
        //[Route("update-pharmacy-purchase-bill")]
        //public async Task<ActionResult> UpdatePharmacyPurchaseBillAsync([FromBody] PharmacyPurchaseBill model)
        //{
        //    model = (PharmacyPurchaseBill)EmptyFilter.Handler(model);
        //    try
        //    {
        //        var response = await this.pharmacyService.UpdatePharmacyPurchaseBillAsync(model);
        //        return this.Success(response);
        //    }
        //    catch (Exception e)
        //    {
        //        return this.ServerError(e.Message);
        //    }

        //    //var productModel = new ProductModel
        //    //{
        //    //    PharmacyProductId = model.PharmacyProductId
        //    //};
        //    //var pharmacyProduct = (await this.pharmacyService.FetchPharmacyProduct(productModel)).ToList();
        //    ////var pharmacyProduct = await this.pharmacyService.FindPharmacyProductByPharmacyProductId(model.PharmacyProductId);
        //    //var pharmacyPurchaseHeader = (await this.pharmacyService.FetchAddedPurchaseBillAsync(model.PharmacyPurchaseHeaderId, null)).ToList();


        //    //var pharmacyLogModel = new PharmacyLogModel
        //    //{
        //    //    AccountId = model.LoginAccountId,
        //    //    PharmacyLogTypeId = (int)PharmacyLogTypes.Pharmacy_Purchase,
        //    //    LogFrom = (short)model.LoginRoleId,
        //    //    LogDate = DateTime.Now,
        //    //    LogDescription = $@"Purchase Quantity Updated Successfully<br/>Bill Number:'{pharmacyPurchaseHeader[0].BillNumber}<br/>Product Name:'{pharmacyProduct[0].ProductName}'<br/>Quantity Updated:'{model.Quantity}'"
        //    //};
        //    //await this.pharmacyLogService.LogAsync(pharmacyLogModel);


        //}

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("pharmacy-bill-reports")]
        [ProducesResponseType(200, Type = typeof(List<IndentInventoryModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPharmacyBillReportsAsync([FromBody] IndentInventoryFilterModel model, [FromHeader] LocationHeader location)
        {
            model = (IndentInventoryFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            if (!string.IsNullOrEmpty(model.EncryptedPatientId))
            {
                model.PatientId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedPatientId));
            }

            var receiptReports = await this.reportServices.FetchPharmacyBillReportsAsync(model);

            var receiptReportsModels = receiptReports.ToList();
            if (!string.IsNullOrEmpty(model.EncryptedPatientId) && receiptReportsModels.ToArray().Length == 0)
            {
                receiptReports = new List<PharmacyBillReportModel> { new PharmacyBillReportModel { PatientId = Convert.ToInt32(model.PatientId) } };
                return this.Success(receiptReports);
            }

            foreach (var item in receiptReportsModels)
            {
                item.EncryptedPatientId = this.aesHelper.Encode(item.PatientId.ToString());
            }
            return this.Success(receiptReports);
        }
        /// <summary>
        /// To fetch list of products.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("pharmacy-stock-reports")]
        [ProducesResponseType(200, Type = typeof(List<PharamacyReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPharamcyStockReportAsync([FromBody] IndentInventoryFilterModel model)
        {
            model = (IndentInventoryFilterModel)EmptyFilter.Handler(model);
            var inPatients = await this.reportServices.FetchPharamcyStockReportAsync(model);
            return this.Success(inPatients);
        }

        /// <summary>
        /// The fetch pharmacy retail stock report async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Authorize]
        [Route("fetch-pharmacy-retail-stock-report")]
        public async Task<ActionResult> FetchPharmacyRetailStockReportAsync([FromBody] PharmacyStockModel model, [FromHeader] LocationHeader location)
        {
            var models = model ?? new PharmacyStockModel();
            models.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var response = await this.reportServices.FetchPharmacyRetailStockReportAsync(models);
            return this.Success(response);
        }

        /// <summary>
        /// The fetch pharmacy stock report async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Authorize]
        [Route("fetch-pharmacy-stock-report")]
        public async Task<ActionResult> FetchPharmacyStockReportAsync([FromBody] PharmacyStockModel model, [FromHeader] LocationHeader location)
        {
            var models = model ?? new PharmacyStockModel();
            models.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var response = await this.reportServices.FetchPharmacyStockReportAsync(models);
            return this.Success(response);
        }
        /// <summary>
        /// To fetch list of doctor revenue.
        /// </summary>
        /// <param name="model">
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of employee revenue.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("doctor-revenue-reports")]
        [ProducesResponseType(200, Type = typeof(List<EmployeeRevenueReportsFilterModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDoctorRevenueReportsAsync([FromBody] EmployeeRevenueReportsFilterModel model, [FromHeader] LocationHeader header)
        {

            model = (EmployeeRevenueReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var doctorRevenuereports = await this.reportServices.FetchDoctorRevenueReportsAsync(model);
            return this.Success(doctorRevenuereports);
        }

        /// <summary>
        /// The fetch all pending indent.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("pharmacy-indent-issue-reports")]
        public async Task<ActionResult> FetchPharmacyIndentIssueReportsAsync([FromBody] PharmacyIndentHeaderModel model, [FromHeader] LocationHeader location)
        {
            model = (PharmacyIndentHeaderModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var response = await this.reportServices.FetchPharmacyIndentIssueReportsAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// The fetch added purchase bill async.
        /// </summary>
        /// <param name="model">
        /// The pharmacy header id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>

        [HttpPost]
        [Route("send-final-bill-email")]
        public async Task<ActionResult> SendFinalBillEmailAsync([FromBody] IndentInventoryFilterModel model)
        {
            model = (IndentInventoryFilterModel)EmptyFilter.Handler(model);
            var resp = new IndentInventoryFilterModel();
            var res = new IndentInventoryFilterModel();
            if (model.FromDate == null)
            {
                resp = await this.reportServices.FetchFromDateAsync();

            }
            if (model.ToDate == null)
            {
                res = await this.reportServices.FetchToDateAsync();

            }
            //var email = await this.resourceService.FetchPharmaUsers();
            if (this.runningEnvironment.CurrentEnvironment != "Local")
            {
                var userList = new List<AccountModel> { };

                if (this.runningEnvironment.CurrentEnvironment == "Production")
                {
                    userList.Add(new AccountModel { Email = "director@maasharada.com", FullName = "Director" });
                    userList.Add(new AccountModel { Email = "purchase@maasharada.com", FullName = "Purchase" });
                    userList.Add(new AccountModel { Email = "pharmacy@maasharada.com", FullName = "Pharmacy" });
                }
                else
                {
                    // userList.Add(new AccountModel { Email = "vmanish@docassistant.net", FullName = "Manish" });
                    userList.Add(new AccountModel { Email = "spchandana@sujainfo.net", FullName = "Priya Chandana" });
                    userList.Add(new AccountModel { Email = "ksatyavani@sujainfo.net", FullName = "Satyavani" });
                    userList.Add(new AccountModel { Email = "puday@sujainfo.net", FullName = "Uday" });
                }
                var mailTemplateModel = new PharmacyBillReportModel
                {
                    CashTotal = model.CashTotal,
                    CardTotal = model.CardTotal,
                    GrossTotal = model.GrossTotal,
                    GstTotal = model.GstTotal,
                    FromDate = model.FromDate == null ? resp.SaleDate : model.FromDate,
                    ToDate = model.ToDate == null ? res.SaleDate : model.ToDate,
                    pharmacyBillType = model.pharmacyBillType

                };

                var emailResponse = await this.emailSender.SendFinalBillEmailAsync(userList, mailTemplateModel);
            }
            return this.Success("Email sent successfully.");
        }

        /// <summary>
        /// The fetch added purchase bill async.
        /// </summary>
        /// <param name="model">
        /// The pharmacy header id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>

        //[HttpPost]
        //[Route("product-profit-report")]
        //public async Task<ActionResult> FetchProductProfitReport([FromBody] PharmacyProductProfit model)
        //{
        //    try
        //    {
        //        model = (PharmacyProductProfit)EmptyFilter.Handler(model);
        //        var response = await this.reportServices.FetchProductProfitReportAsync(model);
        //        return this.Success(response);
        //    }
        //    catch (Exception e)
        //    {
        //        return null;
        //    }
        //}

        /// <summary>
        /// Fetches the patient registration report.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("patient-registration-report")]
        public async Task<ActionResult> FetchPatientRegistrationReport([FromBody] RegistrationBillModel model, [FromHeader] LocationHeader location)
        {
            model ??= new RegistrationBillModel();
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var response = await this.reportServices.FetchPatientRegistrationCount(model);
            return this.Success(response);
        }

        /// <summary>
        /// To fetch list of doctor revenue.
        /// </summary>
        /// <param name="model">
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of employee revenue.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("doctor-revenue-location-reports")]
        [ProducesResponseType(200, Type = typeof(List<EmployeeRevenueReportsFilterModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDoctorRevenueLocationReportsAsync([FromBody] EmployeeRevenueReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (EmployeeRevenueReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var doctorRevenuereports = await this.reportServices.FetchDoctorNewRevenueReportsAsync(model);
            return this.Success(doctorRevenuereports);
        }

        /// <summary>
        /// To fetch list of employee revenue.
        /// </summary>
        /// <param name="model">
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of employee revenue.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("employee-revenue-location-reports")]
        [ProducesResponseType(200, Type = typeof(List<EmployeeRevenueReportsFilterModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchEmployeeRevenueLocationReportsAsync([FromBody] EmployeeRevenueReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (EmployeeRevenueReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var employeeRevenueReports = await this.reportServices.FetchEmployeeNewRevenueReportsAsync(model);
            return this.Success(employeeRevenueReports);
        }

        /// <summary>
        /// To fetch list of cancelled appointment .
        /// </summary>
        /// <param name="model">
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of employee revenue.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("cancel-apppointments-with-refund-details")]
        [ProducesResponseType(200, Type = typeof(List<CancelAppointmentReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchCancelledAppointmentReportsAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            //  model = (CancelAppointmentReport)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var response = await this.reportServices.FetchCancelledAppointmentReportsAsync(model);
            if (response != null)
            {
                foreach (var item in response)
                {
                    try
                    {
                        DateTime d1 = DateTime.Today + item.AppointmentTime;
                        item.AppointmentTimeString = d1.ToString("hh:mm tt");
                    }
                    catch (Exception e)
                    {
                        return null;
                    }
                }
            }

            return this.Success(response);
        }

        /// <summary>
        /// To fetch list of cancelled appointment .
        /// </summary>
        /// <param name="model">
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of employee revenue.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("patient-apppointments")]
        [ProducesResponseType(200, Type = typeof(List<CancelAppointmentReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientAppointmentReportsAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            // model.LocationId = Convert.ToInt32(header.LocationId);
            var response = await this.reportServices.FetchPatientAppointmentReportsAsync(model);
            foreach (var item in response)
            {
                try
                {
                    DateTime d1 = DateTime.Today + item.AppointmentTime;
                    item.AppointmentTimeString = d1.ToString("hh:mm tt");
                }
                catch (Exception e)
                {
                    return null;
                }
            }
            return this.Success(response);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("transaction-report")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchTransactionReportsAsync([FromBody] TransactionReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (TransactionReportsFilterModel)EmptyFilter.Handler(model);
            // model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            model.LocationId = model.LocationId > 0 ? model.LocationId : !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchTransactionReportsAsync(model);
            return this.Success(receiptReports);
        }

        /// <summary>
        /// To fetch list of patient pharmacy.
        /// </summary>
        /// <param name="model">
        /// The patient pharmacy model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patient pharmacy sales.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("patient-pharmacy-reports-android")]
        [ProducesResponseType(200, Type = typeof(List<PharmacySaleBill>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientPharmacyReporAndroidtAsync([FromBody] PatienSaleBill1 model)
        {

            model = (PatienSaleBill1)EmptyFilter.Handler(model);

            var patientPharmacyReports = await this.reportServices.FetchPatientPharmacyReportAndroidAsync(model);
            return this.Success(patientPharmacyReports);
        }

        /// <summary>
        /// To fetch list of patient pharmacy.
        /// </summary>
        /// <param name="model">
        /// The patient pharmacy model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patient pharmacy sales.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("new-patient-report")]
        [ProducesResponseType(200, Type = typeof(List<PharmacySaleBill>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchNewPatientAsync([FromBody] OutPatientFilterModel model)
        {

            model = (OutPatientFilterModel)EmptyFilter.Handler(model);

            var newPatientReports = await this.reportServices.FetchNewPatientAsync(model);
            foreach (var item in newPatientReports)
            {
                if (item.AppointmentTimes != null)
                {
                    DateTime d1 = Convert.ToDateTime(item.AppointmentTimes);
                    item.AppointmentTimeString = d1.ToString("hh:mm tt");
                }
            }
            return this.Success(newPatientReports);
        }

        /// <summary>
        /// To fetch list of module wise transaction.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("detailed-revenue-report")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDetailEmployeeRevenueAsync([FromBody] ReceiptReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (ReceiptReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchDetailEmployeeRevenueAsync(model);
            return this.Success(receiptReports);
        }

        /// <summary>
        /// To fetch list of module wise transaction.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("cancel-refund-receipt")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchCancelRefundReportAsync([FromBody] ReceiptReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (ReceiptReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchCancelRefundReportAsync(model);
            return this.Success(receiptReports);
        }

        /// <summary>
        /// To fetch list of module wise transaction.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("op-daily-gross-collection")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchOPDailyGrossReportAsync([FromBody] ReceiptReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (ReceiptReportsFilterModel)EmptyFilter.Handler(model);
            //  model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            model.LocationId = model.LocationId > 0 ? model.LocationId : !string.IsNullOrEmpty(header.LocationId) ? Convert.ToInt32(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchOPDailyGrossReportAsync(model);
            return this.Success(receiptReports);
        }

        /// <summary>
        /// To fetch list of module wise transaction.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("login-userwise-collection")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLoginUserCollectionReportAsync([FromBody] ReceiptReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (ReceiptReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchLoginUserCollectionReportAsync(model);
            return this.Success(receiptReports);
        }

        /// <summary>To fetch list of bill daily report.</summary>
        /// <param name="model"></param>
        /// <param name="header"></param>
        /// <returns>The <see cref="Task"/>.</returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("bill-daily-report")]
        [ProducesResponseType(200, Type = typeof(List<BillDailyReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchBillDailyReportAsync([FromBody] ReceiptReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (ReceiptReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var billReports = await this.reportServices.FetchBillDailyReportAsync(model);
            return this.Success(billReports);
        }

        /// <summary>
        /// To fetch list of module wise transaction.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("user-collection")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchUserCollectionReportAsync([FromBody] ReceiptReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (ReceiptReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchUserCollectionReportAsync(model);
            return this.Success(receiptReports);
        }

        /// <summary>
        /// To fetch list of module wise transaction.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("providers-appointment-slot")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScheduleAppointmentReportAsync([FromBody] ReceiptReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (ReceiptReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;

            SlotRequest model1 = new SlotRequest();
            model1.ConsultationTypeId = model.ConsultationTypeId;
            model1.ProviderId = (int)model.ProviderId;
            model1.ProviderAvailabilityId = (int)model.ProviderAvailabilityId;
            // model1.SlotDate = DateTime.Now.ToString("yyyy-MM-dd");
            model1.SlotDate = model.FromDate;
            model1.SpecializationId = (int)model.SpecializationId;
            model1.LocationId = model.LocationId;
            model1.PatientId = (int)model.PatientId;
            model1.TimeZone = model.TimeZone;
            model1.Offset = model.Offset;

            var data = await this.providerLocationServices.FetchNewSlots(model1);
            var receiptReports = await this.reportServices.FetchScheduleAppointmentReportAsync(model);
            // var bookSlot = receiptReports.ToList();

            var finalSlotList = new List<DoctorScheduleSlotModel>();
            foreach (var item in data)
            {
                DoctorScheduleSlotModel finalSlot = new DoctorScheduleSlotModel();
                finalSlot.SlotName = item.SlotName;
                finalSlot.SlotTime = item.SlotTime;
                finalSlot.SlotName12HoursEnd = item.SlotName12HoursEnd;
                finalSlot.Status = item.Status;
                finalSlot.SlotType = item.SlotType;

                finalSlotList.Add(finalSlot);
            }
            foreach (var item in receiptReports.ToList())
            {
                // finalSlotList.Find(m => m.SlotName == item.AppointmentTime);
                foreach (var target in finalSlotList)
                {
                    if (item.AppointmentTime == target.SlotName)
                    {
                        target.AppointmentNo = item.AppointmentNo;
                        target.Mobile = item.Mobile;
                        target.PatientMobile = item.PatientMobile;
                        target.PatientName = item.PatientName;
                        target.ProviderName = item.ProviderName;
                        target.UMRNo = item.UMRNo;
                        target.AppointmentTime = item.AppointmentTime;
                        target.CreatedDate = item.CreatedDate;
                        target.CreatedByName = item.CreatedByName;
                    }

                }
            }
            finalSlotList[0].TotalItems = receiptReports.ToList()[0].TotalItems;
            var finalSlotListSorted = finalSlotList.OrderBy(x => x.SlotTime);
            return this.Success(finalSlotListSorted);

        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("all-appointments")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllAppointmentsAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            model.ProviderId = await this.accountService.GetReferenceAsync(model.AccountId);
            var inPatients = await this.reportServices.FetchAllAppointments(model);
            foreach (var item in inPatients)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
            }
            return this.Success(inPatients);
        }
        [HttpPost]
        [Authorize]
        [Route("fetch-reminders")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRemindersAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            model.ProviderId = await this.accountService.GetReferenceAsync(model.AccountId);
            var inPatients = await this.reportServices.FetchRemindersAsync(model);
            foreach (var item in inPatients)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
            }
            return this.Success(inPatients);
        }
        [HttpPost]
        [Authorize]
        [Route("all-admissions")]
        [ProducesResponseType(200, Type = typeof(List<OutReport>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllAdmissionAsync([FromBody] OutPatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (OutPatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            model.ProviderId = await this.accountService.GetReferenceAsync(model.AccountId);
            var inPatients = await this.reportServices.FetchAllAdmission(model);
            foreach (var item in inPatients)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
            }
            return this.Success(inPatients);
        }
        /// <summary>
        /// To fetch list of module wise transaction.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("admission-form-report")]
        public async Task<ActionResult> FetchAdmissionFormReportAsync([FromBody] PatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (PatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchAdmissionFormReportAsync(model);
            return this.Success(receiptReports);
        }

        /// <summary>
        /// To fetch list of module wise transaction.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("scheduledScanSlots")]
        [ProducesResponseType(200, Type = typeof(List<FetchBookScanAppointment>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScheduleScanReportAsync([FromBody] ReceiptReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (ReceiptReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;

            var bookedslot = await this.reportServices.FetchScheduleScanReportAsync(model);

            ScanMachineAvailabilityBasedOnDaysModel model1 = new ScanMachineAvailabilityBasedOnDaysModel();
            model1.ScanMachineMasterId = model.Id;
            model1.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;

            var allslot = await this.scanMachineAvailabilityService.FetchMachineSlotsAsync(model1);
            // return this.Success(bookedslot);

            var finalSlotList = new List<DoctorScheduleSlotModel>();
            foreach (var item in allslot)
            {
                DoctorScheduleSlotModel finalSlot = new DoctorScheduleSlotModel();
                finalSlot.SlotName = item.SlotName;
                finalSlot.SlotTime = item.SlotTime;
                finalSlot.SlotName12HoursEnd = item.SlotName12HoursEnd;
                finalSlot.Status = item.Status;
                finalSlot.SlotType = item.SlotType;
                finalSlot.EndTime = item.EndTime;
                finalSlotList.Add(finalSlot);
            }
            foreach (var item in bookedslot.ToList())
            {

                foreach (var target in finalSlotList)
                {
                    if (item.AppointmentTime == target.SlotName)
                    {
                        target.AppointmentNo = item.RequisitionNumber;
                        target.Mobile = item.Mobile;
                        target.PatientMobile = item.Mobile;
                        target.PatientName = item.PatientName;
                        target.ProviderName = item.MachineName;
                        target.UMRNo = item.UMRNO;
                        target.AppointmentTime = item.AppointmentTime;
                        target.AppointmentEndTime = item.AppointmentEndTime;
                        target.CreatedDate = (DateTime)item.CreatedDate;
                        target.CreatedByName = item.CreatedByName;
                        target.ScanTestName = item.ScanTestName;

                    }

                }
            }
            finalSlotList[0].TotalItems = bookedslot.ToList()[0].TotalItems;
            var finalSlotListSorted = finalSlotList.OrderBy(x => x.SlotTime);
            return this.Success(finalSlotListSorted);

        }



        /// <summary>
        /// To fetch mis report.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("mis-fetch-report")]
        public async Task<ActionResult> FetchMISReportAsync([FromBody] FilterModel model, [FromHeader] LocationHeader header)
        {
            //  model.LocationId = model.LocationId > 0 ? model.LocationId : !string.IsNullOrEmpty(header.LocationId) ? Convert.ToInt32(header.LocationId) : (int?)null;

            // model = (PatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = model.LocationId > 0 ? model.LocationId : (int?)null;
            var receiptReports = await this.reportServices.FetchMISReportAsync(model);
            return this.Success(receiptReports);
        }


        /// <summary>
        /// To fetch list of module wise transaction.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("new-doctor-detail-revenue")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchNewDoctorDetailRevenueReportAsync([FromBody] ReceiptReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (ReceiptReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchNewDoctorDetailRevenueReportAsync(model);
            return this.Success(receiptReports);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="model"></param>
        /// <param name="header"></param>
        /// <returns></returns>
        [HttpPost]
        [Authorize]
       [Route("lab-tat-new-report")]
       [ProducesResponseType(200, Type = typeof(List<LabTATNewReportModel>))]
       [ProducesResponseType(500)]
        public async Task<ActionResult> FetchNewDoctorDetailRevenueReportAsync([FromBody] LabTATNewReportModel model)
        {
           // model = (LabBookingModel)EmptyFilter.Handler(model);
            //model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var newTATReports = await this.reportServices.FetchLabNewTATReportAsync(model);
            return this.Success(newTATReports);
        }

        [HttpPost]        
        [Authorize]
        [Route("lab-monthly-count")]
        [ProducesResponseType(200, Type = typeof(List<LabMothlyCountModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLabMonthlyCountReportAsync([FromBody] LabMothlyCountModel model)
        {
            // model = (LabBookingModel)EmptyFilter.Handler(model);
            //model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var labCountReports = await this.reportServices.FetchLabMonthlyCountReportAsync(model);
            return this.Success(labCountReports);
        }

        [HttpPost]        
        [Authorize]
        [Route("consultation-wise-report")]
        [ProducesResponseType(200, Type = typeof(List<ConsultantWiseRevenueModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchConsultantWiseRevenueReportAsync([FromBody] ConsultantWiseRevenueModel model)
        {
            // model = (LabBookingModel)EmptyFilter.Handler(model);
           // model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var consultantWiseReports = await this.reportServices.ConsultantWiseRevenue(model);
            return this.Success(consultantWiseReports);
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="model"></param>
        /// <param name="header"></param>
        /// <returns></returns>
        [HttpPost]
        [Authorize]
        [Route("admission-department")]
        [ProducesResponseType(200, Type = typeof(List<LabTATNewReportModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAdmissionDepartmentAsync([FromBody] PatientFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (PatientFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchAdmissionDepartmentAsync();
            return this.Success(receiptReports);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="model"></param>
        /// <param name="header"></param>
        /// <returns></returns>
        [HttpPost]
        [Authorize]
        [Route("consultant-statistics")]
        [ProducesResponseType(200, Type = typeof(List<LabTATNewReportModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchConsultationCountAsync([FromBody] EmployeeRevenueReportsFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (EmployeeRevenueReportsFilterModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var receiptReports = await this.reportServices.FetchConsultationCountAsync(model);
            return this.Success(receiptReports);
        }
        /// <summary>
        /// To fetch list of module wise transaction.
        /// </summary>
        /// <param name="model">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("medication-count-report")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchMedicationCountReportAsync([FromBody] MedicineRankFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (MedicineRankFilterModel)EmptyFilter.Handler(model);
            //model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var receiptReports = await this.reportServices.FetchProductSaleCountReport(model);
            return this.Success(receiptReports);
        }
        
        [HttpPost]
        [Authorize]
        [Route("lab-detail-report")]
        [ProducesResponseType(200, Type = typeof(List<LabBookingModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchMedicationCountReportAsync([FromBody] LabBookingModel model, [FromHeader] LocationHeader header)
        {
            model = (LabBookingModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var receiptReports = await this.reportServices.FetchLabDetailReports(model);
            return this.Success(receiptReports);
        }
        [HttpPost]
        [Authorize]
        [Route("expiry-stock-report")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchExpiredStockAsync([FromBody] PharmacyRetailStockReportModel model, [FromHeader] LocationHeader header)
        {
           // model = (PharmacyRetailStockReportModel)EmptyFilter.Handler(model);
           // model.LocationId = model.LocationId != 0 ? model.LocationId : !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var response = await this.reportServices.FetchExpiredStockAsync(model);

            return this.Success(response);
        }

        [HttpPost]
        [Authorize]
        [Route("bill-discount")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchBillDiscountAsync([FromBody] BillDiscountModel model)
        {
            model = (BillDiscountModel)EmptyFilter.Handler(model);
            // model.LocationId = model.LocationId != 0? model.LocationId: !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var response = await this.reportServices.FetchBillDiscountAsync(model);
            return this.Success(response);
        }


        [HttpPost]
        [Authorize]
        [Route("dept-wise-admission-count")]
        [ProducesResponseType(200, Type = typeof(List<AdmissionCountsFilterModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAdmissionCountDeptWiseAsync([FromBody] AdmissionCountsFilterModel model)
        {
            model = (AdmissionCountsFilterModel)EmptyFilter.Handler(model);
            var response = await this.reportServices.FetchAdmissionCountDeptWiseAsync(model);
            return this.Success(response);
        }

        [HttpPost]
        [Authorize]
        [Route("retail-indent-report")]
        [ProducesResponseType(200, Type = typeof(List<ReceiptReportsModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRetailIndentIssue([FromBody] RetailIndentIssueModel model, [FromHeader] LocationHeader header)
        {          
            var response = await this.reportServices.FetchRetailIndentIssue(model);
            return this.Success(response);        
        }
        [HttpPost]
        [Authorize]
        [Route("avg-stay-ward-wise-report")]
        [ProducesResponseType(200, Type = typeof(List<AdmissionCountsFilterModel>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAvarageStayWardWiseReportAsync([FromBody] AdmissionCountsFilterModel model)
        {
            model = (AdmissionCountsFilterModel)EmptyFilter.Handler(model);
            var response = await this.reportServices.FetchAvarageStayWardWiseReportAsync(model);
            return this.Success(response);
        }


        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("call-center-report")]
        [ProducesResponseType(typeof(List<Call_HistoryModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] Call_HistoryModel model, [FromHeader] LocationHeader header)
        {
            model = (Call_HistoryModel)EmptyFilter.Handler(model);
            var beds = await this.reportServices.CallHistoryAsync(model);
            return beds == null ? this.ServerError() : this.Success(beds);
        }
    }
    
}
